import os
import json
from telegram import InlineKeyboardButton, InlineKeyboardMarkup
from datetime import datetime

TICKETS_FILE = "tickets.json"

def load_tickets():
    if not os.path.exists(TICKETS_FILE):
        return []
    with open(TICKETS_FILE, "r", encoding="utf-8") as f:
        return json.load(f)

def save_tickets(tickets):
    with open(TICKETS_FILE, "w", encoding="utf-8") as f:
        json.dump(tickets, f, indent=2)

def get_new_ticket_count():
    tickets = load_tickets()
    return sum(1 for t in tickets if t["status"] == "open")

def get_ticket_by_id(ticket_id):
    tickets = load_tickets()
    for t in tickets:
        if t["id"] == ticket_id:
            return t
    return None

def update_ticket(ticket):
    tickets = load_tickets()
    for i, t in enumerate(tickets):
        if t["id"] == ticket["id"]:
            tickets[i] = ticket
            save_tickets(tickets)
            return
    tickets.append(ticket)
    save_tickets(tickets)

def add_ticket_reply(ticket_id, sender, text):
    tickets = load_tickets()
    for t in tickets:
        if t["id"] == ticket_id:
            t["messages"].append({
                "from": sender,
                "text": text,
                "time": datetime.now().strftime("%m-%d-%Y %I:%M %p")
            })
            save_tickets(tickets)
            return

async def user_ticket_menu(update, context):
    user_id = update.effective_user.id
    tickets = [t for t in load_tickets() if t["user_id"] == user_id]
    if not tickets:
        await update.callback_query.edit_message_text(
            "You have no support tickets.\n\nPress 'New Ticket' to create one.",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("➕ New Ticket", callback_data="ticket_new")],
                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
            ])
        )
        return
    buttons = [
        [InlineKeyboardButton(
        f"{'❌' if t['status']=='closed' else ('🔴' if t['priority']=='High' else '🟡' if t['priority']=='Medium' else '🟢')} {t['subject']} ({t['status']})",
        callback_data=f"ticket_view_{t['id']}"
    ) for t in tickets]
    ]
    buttons.append([InlineKeyboardButton("➕ New Ticket", callback_data="ticket_new")])
    buttons.append([InlineKeyboardButton("🆘 Support Tickets", callback_data="support_tickets")])
    await update.callback_query.edit_message_text(
        "Your Support Tickets:",
        reply_markup=InlineKeyboardMarkup(buttons)
    )

async def user_ticket_view(update, context, ticket_id):
    ticket = get_ticket_by_id(ticket_id)
    if not ticket:
        await update.callback_query.edit_message_text("Ticket not found.")
        return
    priority_emojis = {"High": "🔴", "Medium": "🟡", "Low": "🟢"}
    priority_emoji = priority_emojis.get(ticket['priority'], "🟢")
    text = f"Subject: <b>{ticket['subject']}</b>\nPriority: {priority_emoji} <b>{ticket['priority']}</b>\nStatus: <b>{ticket['status'].capitalize()}</b>\n\n"
    for msg in ticket["messages"]:
        who = "👤 You" if msg["from"] == "user" else "🛠️ Admin"
        text += f"{who} ({msg['time']}):\n{msg['text']}\n\n"
    buttons = []
    if ticket["status"] == "open":
        buttons.append([
            InlineKeyboardButton("💬 Reply", callback_data=f"ticket_reply_{ticket_id}"),
            InlineKeyboardButton("✅ Close", callback_data=f"ticket_close_{ticket_id}")
        ])
    buttons.append([InlineKeyboardButton("🔄 Refresh", callback_data=f"ticket_refresh_{ticket_id}")])
    buttons.append([InlineKeyboardButton("⬅️ Back to Tickets", callback_data="ticket_menu")])
    buttons.append([InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")])
    await update.callback_query.edit_message_text(
        text=text,
        reply_markup=InlineKeyboardMarkup(buttons),
        parse_mode="HTML"
    )

async def user_ticket_new_start(update, context):
    context.user_data["new_ticket"] = {}
    context.user_data["awaiting_ticket_subject"] = True
    await update.callback_query.edit_message_text(
        "Please enter a subject for your support ticket (text only):",
        reply_markup=InlineKeyboardMarkup([
            [InlineKeyboardButton("⬅️ Cancel", callback_data="ticket_menu")]
        ])
    )

async def handle_user_message(update, context):
    user_id = update.effective_user.id
    username = update.effective_user.username or update.effective_user.full_name

    # New ticket subject
    if context.user_data.get("awaiting_ticket_subject"):
        subject = update.message.text.strip()
        if not subject:
            await update.message.reply_text("Subject cannot be empty. Please enter a subject (text only):")
            return
        context.user_data["new_ticket"]["subject"] = subject
        context.user_data["awaiting_ticket_subject"] = False
        context.user_data["awaiting_ticket_message"] = True
        await update.message.reply_text(
            "Please describe your issue (text only, no images/files allowed):"
        )
        return

    # New ticket message
    if context.user_data.get("awaiting_ticket_message"):
        text = update.message.text.strip()
        if not text:
            await update.message.reply_text("Message cannot be empty. Please describe your issue (text only):")
            return
        context.user_data["new_ticket"]["message"] = text
        context.user_data["awaiting_ticket_message"] = False
        await update.message.reply_text(
            "Select the priority level:",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("🔴 High", callback_data="ticket_priority_high")],
                [InlineKeyboardButton("🟡 Medium", callback_data="ticket_priority_medium")],
                [InlineKeyboardButton("🟢 Low", callback_data="ticket_priority_low")]
            ])
        )
        return

    # Ticket reply
    if context.user_data.get("awaiting_ticket_reply"):
        text = update.message.text.strip()
        if not text:
            await update.message.reply_text("Reply cannot be empty. Please enter your reply (text only):")
            return
        ticket_id = context.user_data.get("reply_ticket_id")
        add_ticket_reply(ticket_id, "user", text)
        context.user_data["awaiting_ticket_reply"] = False
        context.user_data["reply_ticket_id"] = None
        await update.message.reply_text(
            "Reply sent.",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("⬅️ Back to Tickets", callback_data="ticket_menu")]
            ])
        )
        return

async def user_ticket_reply_start(update, context, ticket_id):
    context.user_data["awaiting_ticket_reply"] = True
    context.user_data["reply_ticket_id"] = ticket_id
    await update.callback_query.edit_message_text(
        "Enter your reply (text only):",
        reply_markup=InlineKeyboardMarkup([
            [InlineKeyboardButton("⬅️ Cancel", callback_data=f"ticket_view_{ticket_id}")]
        ])
    )

async def user_ticket_close(update, context, ticket_id):
    ticket = get_ticket_by_id(ticket_id)
    if not ticket or ticket["status"] != "open":
        await update.callback_query.edit_message_text("Ticket not found or already closed.")
        return
    ticket["status"] = "closed"
    update_ticket(ticket)
    await update.callback_query.edit_message_text("Ticket closed.", reply_markup=InlineKeyboardMarkup([
        [InlineKeyboardButton("⬅️ Back to Tickets", callback_data="ticket_menu")]
    ]))

async def handle_priority_selection(update, context, priority):
    ticket = context.user_data.get("new_ticket")
    if not ticket:
        await update.callback_query.edit_message_text("No ticket in progress.")
        return
    ticket_id = str(int(datetime.now().timestamp() * 1000))
    new_ticket = {
        "id": ticket_id,
        "user_id": update.effective_user.id,
        "username": update.effective_user.username or update.effective_user.full_name,
        "subject": ticket["subject"],
        "messages": [{
            "from": "user",
            "text": ticket["message"],
            "time": datetime.now().strftime("%m-%d-%Y %I:%M %p")
        }],
        "priority": priority,
        "status": "open",
        "created": datetime.now().strftime("%m-%d-%Y %I:%M %p"),
        "admin_seen": False
    }
    tickets = load_tickets()
    tickets.append(new_ticket)
    save_tickets(tickets)
    context.user_data["new_ticket"] = None
    await update.callback_query.edit_message_text(
        f"Ticket created with priority {priority}. An admin will respond soon.",
        reply_markup=InlineKeyboardMarkup([
            [InlineKeyboardButton("⬅️ Back to Tickets", callback_data="ticket_menu")]
        ])
    )

# --- Admin Side ---

async def admin_ticket_menu(update, context):
    tickets = load_tickets()
    if not tickets:
        await update.callback_query.edit_message_text(
            "No support tickets found.",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("⬅️ Back to Admin Panel", callback_data="admin_panel")]
            ])
        )
        return
    buttons = [
        [InlineKeyboardButton(
        f"{'❌' if t['status']=='closed' else ('🔴' if t['priority']=='High' else '🟡' if t['priority']=='Medium' else '🟢')} {t['subject']} ({t['status']})",
        callback_data=f"admin_ticket_view_{t['id']}"
    ) for t in tickets]
    ]
    buttons.append([InlineKeyboardButton("⬅️ Back to Admin Panel", callback_data="admin_panel")])
    await update.callback_query.edit_message_text(
        "All Support Tickets:",
        reply_markup=InlineKeyboardMarkup(buttons)
    )

async def admin_ticket_view(update, context, ticket_id):
    ticket = get_ticket_by_id(ticket_id)
    if not ticket:
        await update.callback_query.edit_message_text("Ticket not found.")
        return
    priority_emojis = {"High": "🔴", "Medium": "🟡", "Low": "🟢"}
    priority_emoji = priority_emojis.get(ticket['priority'], "🟢")
    username = ticket.get('username')
    if username and not username.startswith("id:"):
        user_link = f'<a href="https://t.me/{username}">@{username}</a>'
    else:
        user_link = username or "Unknown"
    text = (
        f"Subject: <b>{ticket['subject']}</b>\n"
        f"Priority: {priority_emoji} <b>{ticket['priority']}</b>\n"
        f"Status: <b>{ticket['status'].capitalize()}</b>\n"
        f"User: {user_link}\n\n"
    )
    for msg in ticket["messages"]:
        who = "👤 User" if msg["from"] == "user" else "🛠️ Admin"
        text += f"{who} ({msg['time']}):\n{msg['text']}\n\n"
    buttons = []
    if ticket["status"] == "open":
        buttons.append([
            InlineKeyboardButton("💬 Reply", callback_data=f"admin_ticket_reply_{ticket_id}"),
            InlineKeyboardButton("✅ Close", callback_data=f"admin_ticket_close_{ticket_id}"),
            InlineKeyboardButton("🗑️ Delete", callback_data=f"admin_ticket_delete_{ticket_id}")
        ])
    else:
        buttons.append([
            InlineKeyboardButton("💬 Reply", callback_data=f"admin_ticket_reply_{ticket_id}"),
            InlineKeyboardButton("♻️ Reopen", callback_data=f"admin_ticket_reopen_{ticket_id}"),
            InlineKeyboardButton("🗑️ Delete", callback_data=f"admin_ticket_delete_{ticket_id}")
        ])
    buttons.append([InlineKeyboardButton("🔄 Refresh", callback_data=f"admin_ticket_refresh_{ticket_id}")])
    buttons.append([InlineKeyboardButton("⬅️ Back to Admin Panel", callback_data="admin_panel")])
    await update.callback_query.edit_message_text(
        text=text,
        reply_markup=InlineKeyboardMarkup(buttons),
        parse_mode="HTML",
        disable_web_page_preview=True
    )

async def admin_ticket_reply_start(update, context, ticket_id):
    context.user_data["awaiting_admin_ticket_reply"] = True
    context.user_data["admin_reply_ticket_id"] = ticket_id
    await update.callback_query.edit_message_text(
        "Enter your reply to the user (text only):",
        reply_markup=InlineKeyboardMarkup([
            [InlineKeyboardButton("⬅️ Cancel", callback_data=f"admin_ticket_view_{ticket_id}")]
        ])
    )

async def handle_admin_message(update, context):
    if context.user_data.get("awaiting_admin_ticket_reply"):
        text = update.message.text.strip()
        if not text:
            await update.message.reply_text("Reply cannot be empty. Please enter your reply (text only):")
            return
        ticket_id = context.user_data.get("admin_reply_ticket_id")
        add_ticket_reply(ticket_id, "admin", text)
        context.user_data["awaiting_admin_ticket_reply"] = False
        context.user_data["admin_reply_ticket_id"] = None
        await update.message.reply_text(
            "Reply sent.",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("⬅️ Back to Tickets", callback_data="admin_ticket_menu")]
            ])
        )
        return

async def admin_ticket_close(update, context, ticket_id):
    ticket = get_ticket_by_id(ticket_id)
    if not ticket or ticket["status"] != "open":
        await update.callback_query.edit_message_text("Ticket not found or already closed.")
        return
    ticket["status"] = "closed"
    update_ticket(ticket)
    await update.callback_query.edit_message_text("Ticket closed.", reply_markup=InlineKeyboardMarkup([
        [InlineKeyboardButton("⬅️ Back to Admin Panel", callback_data="admin_panel")]
    ]))

async def admin_ticket_delete(update, context, ticket_id):
    tickets = load_tickets()
    tickets = [t for t in tickets if t["id"] != ticket_id]
    save_tickets(tickets)
    await update.callback_query.edit_message_text("Ticket deleted.", reply_markup=InlineKeyboardMarkup([
        [InlineKeyboardButton("⬅️ Back to Admin Panel", callback_data="admin_panel")]
    ]))

async def admin_ticket_reopen(update, context, ticket_id):
    ticket = get_ticket_by_id(ticket_id)
    if not ticket or ticket["status"] == "open":
        await update.callback_query.edit_message_text("Ticket not found or already open.")
        return
    ticket["status"] = "open"
    update_ticket(ticket)
    await update.callback_query.edit_message_text("Ticket reopened.", reply_markup=InlineKeyboardMarkup([
        [InlineKeyboardButton("⬅️ Back to Admin Panel", callback_data="admin_panel")]
    ]))